//
//  TodoItemEditor.swift
//  Do It
//
//  Created by Jim Dovey on 10/6/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import SwiftUI

fileprivate extension TodoItem {
    var hasDueDate: Bool {
        get { date != nil }
        set {
            if newValue && date == nil {
                date = Date()
            } else {
                date = nil
            }
        }
    }

    var definiteDate: Date {
        get { date ?? Date() }
        set { date = newValue }
    }
    
    var definiteNotes: String {
        get { notes ?? "" }
        set {
            if newValue.isEmpty {
                notes = nil
            } else {
                notes = newValue
            }
        }
    }
}

struct TodoItemEditor: View {
    @Binding var item: TodoItem
    @EnvironmentObject var data: DataCenter

    @State private var showTime: Bool
    init(item: Binding<TodoItem>) {
        self._item = item
        self._showTime = State(wrappedValue: false)

        if let date = item.date.wrappedValue {
            let components = Calendar.current.dateComponents(
                [.hour, .minute], from: date)
            self.showTime = components.hour! != 0 || components.minute! != 0
        }
    }

    var notesEditor: some View {
        TextView(text: self.$item.definiteNotes)
            .padding(.horizontal)
            .navigationBarTitle("Notes: \(item.title)")
    }

    var body: some View {
        Form {
            TextField("Title", text: $item.title)

            Picker("List", selection: $item.listID) {
                ForEach(data.todoLists) { list in
                    Text(list.name).tag(list.id)
                }
            }

            Picker("Priority", selection: $item.priority) {
                ForEach(TodoItem.Priority.allCases, id: \.self) { priority in
                    Text(LocalizedStringKey(priority.rawValue.capitalized))
                }
            }

            HStack {
                Text("Due Date")
                Spacer()
                Toggle("Has Due Date", isOn: $item.hasDueDate.animation())
                    .labelsHidden()
            }

            if self.item.hasDueDate {
                Toggle("Include Time", isOn: $showTime)
                HStack {
                    Spacer()
                    DatePicker("Due Date", selection: $item.definiteDate,
                               displayedComponents: showTime
                                   ? [.date, .hourAndMinute]
                                   : .date)
                        .datePickerStyle(WheelDatePickerStyle())
                        .labelsHidden()
                    Spacer()
                }
            }
            
            NavigationLink(destination: notesEditor) {
                Text("Notes")
            }
        }
        .navigationBarTitle(Text("Editing: \(item.title)"),
                            displayMode: .inline)
    }
}

struct TodoItemEditor_Previews: PreviewProvider {
    static var previews: some View {
        NavigationView {
            StatefulPreviewWrapper(defaultTodoItems[0]) {
                TodoItemEditor(item: $0)
            }
        }
        .environmentObject(DataCenter())
    }
}
